/*
  Visual Studio project setup
  ////////////////////////////
  1. Add this package's 'redistributables\include' directory to your project's 'Additional Include Directories' project property (under C/C++ -> General)
  2. Add this statement to your source code files: '#include <starmicronics/IOPortConnection.h>'
  3. Add this package's 'redistributables\lib' directory to your project's 'Additional Library Directories' project property (under Linker -> General)
  4. Add 'IOPortConnection.lib' to your project's 'Additional Dependencies' project property (under Linker -> Input)
*/

#ifndef ioport__
#define ioport__

#include <starmicronics/Platform.h>

#ifdef COMPILING_STARIOPORT
#define IOPORTCONNECTION_API DLL_EXPORT
#else
#define IOPORTCONNECTION_API DLL_IMPORT
#endif

typedef struct StarPrinterStatus_
{
  SM_BOOLEAN coverOpen;
  SM_BOOLEAN offline;
  SM_BOOLEAN paperEmpty;
} StarPrinterStatus, StarPrinterStatus_0;

#ifdef __cplusplus
extern "C"
{
#endif

/*
    OpenPort
    --------
    This function opens a connection to the port specified.

    Parameters: portName               - (Windows) string taking on one of these forms:
                                                   "usbprn:Queue Name" which opens the USB Printer Class device associated with the specified printer queue
                                                   "COMx" which opens the real serial port (i.e. 'COM1')
                                                   "LPTx" which opens the real parallel port (i.e. 'LPT1')
                portSettings           - string of the form "9600,n,8,1,c" for serial, "" for all others
                ioRequestTimeoutMillis - millisecond timeout controlling internally used communication APIs (this parameter garantees that all of the below APIs will complete in a bounded amount of time, but does NOT garantee the exact timeout length)
    Returns:    0 on failure
                    or
                pointer to the created port (opaque)
    Notes:      1. In the case of serial, the portSettings string contains the
                   following fields:
                       baud: 115200, 57600, 38400, 19200, 9600, 4800
                       parity: n, e, o
                       data-bits: 8, 7
                       stop-bits: 1
                       flow-ctrl: n, h, c

                2. On Windows, OpenPort will fail and return 0 if another process (or driver) already has the same port
                   opened.  In many cases you will want to call OpenPort in a loop until the other process has
                   finished its usage.  For example:

                   void * port = 0; // port handle (opaque pointer)
                   while ((port = OpenPort(L"COM1", L"9600,n,8,1,c", 5000)) == 0)
                   {
                       if (GetLastError() == ERROR_ALREADY_EXISTS) // ALREADY_EXISTS indicates another process's usage - try again
                       {
                           Sleep(100); // suspend the thread - save the earth

                           continue; // try again
                       }
                       else
                       {
                           return EXIT_FAILURE; // this is a different kind of failure - dead
                       }
                   }
*/
#if defined(_WINDOWS) || defined(WIN32)

  IOPORTCONNECTION_API void * CALL_CONVENT OpenPortW(wchar_t const * portName, wchar_t const * portSettings, UINT32 ioRequestTimeoutMillis);

  IOPORTCONNECTION_API void * CALL_CONVENT OpenPortA(char const * portName, char const * portSettings, UINT32 ioRequestTimeoutMillis);

  #ifdef _UNICODE

    #define OpenPort(portName, portSettings, ioRequestTimeoutMillis) OpenPortW(portName, portSettings, ioRequestTimeoutMillis)

  #else

    #define OpenPort(portName, portSettings, ioRequestTimeoutMillis) OpenPortA(portName, portSettings, ioRequestTimeoutMillis)

  #endif

#else

  IOPORTCONNECTION_API void * CALL_CONVENT OpenPort(char const * portName, char const * portSettings, UINT32 ioRequestTimeoutMillis);

#endif

/*
    ClosePort
    --------
    This function closes a connection to the port specified.

    Parameters: port - pointer to a previously created port
    Returns:    SM_FALSE on failure
                  or
                SM_TRUE on success
*/
IOPORTCONNECTION_API SM_BOOLEAN CALL_CONVENT ClosePort(void * port);

/*
    ReadPort
    --------
    This function reads data from the device.

    Parameters: port             - pointer to a previously created port
                readBuffer       - pointer to a byte buffer into which data is read
                size             - size in bytes to read
                sizeCommunicated - out variable into which the actual read size is written
    Returns:    SM_FALSE on failure
                  or
                SM_TRUE on success
    Notes:      Under some interface types, this function will return SM_TRUE even when no data was read in.  Your application should call
                this function a limited number of times until the expected data has been read in or until an application determined
                retry threshold has been reached.
*/
IOPORTCONNECTION_API SM_BOOLEAN CALL_CONVENT ReadPort(void * port, UCHAR * readBuffer, UINT32 size, UINT32 * sizeCommunicated);

/*
    WritePort
    --------
    This function writes data to the device.

    Parameters: port             - pointer to a previously created port
                writeBuffer      - pointer to a byte buffer containing the write data
                size             - size in bytes to write
                sizeCommunicated - out variable into which the actual write size is written
    Returns:    SM_FALSE on failure
                  or
                SM_TRUE on success
    Notes:      Under some interface types, this function will return SM_TRUE even when no data was written out.  Your application should call
                this function a limited number of times until all the data has been written out or until an application determined
                retry threshold has been reached.
*/
IOPORTCONNECTION_API SM_BOOLEAN CALL_CONVENT WritePort(void * port, UCHAR const * writeBuffer, UINT32 size, UINT32 * sizeCommunicated);

/*
    GetOnlineStatus
    --------
    This function retreives the device's online status.

    Parameters: port         - pointer to a previously created port
                onlineStatus - out variable into which the device's online status is written (SM_FALSE = offline, SM_TRUE = online)
    Returns:    SM_FALSE on failure
                  or
                SM_TRUE on success
*/
IOPORTCONNECTION_API SM_BOOLEAN CALL_CONVENT GetOnlineStatus(void * port, SM_BOOLEAN * onlineStatus);

/*
    GetParsedStatusEx
    --------
    This function retreives the device's detailed status.

    Parameters: port              - pointer to a previously created port
                starPrinterStatus - pointer to a StarPrinterStatus_n structure where the devices detailed status is written (structure is StarPrinterStatus_0)
                level             - integer designating the level of status structure (level is 0)
    Returns:    SM_FALSE on failure
                  or
                SM_TRUE on success
*/
IOPORTCONNECTION_API SM_BOOLEAN CALL_CONVENT GetParsedStatusEx(void * port, void * starPrinterStatus, UINT32 level);

/*
    BeginCheckedBlockEx
    --------
    This function initiates a checked block printing operation and returns the devices detailed status.

    Parameters: port              - pointer to a previously created port
                starPrinterStatus - pointer to a StarPrinterStatus_n structure where the devices detailed status is written (structure is StarPrinterStatus_0)
                level             - integer designating the level of status structure (level is 0)
    Returns:    SM_FALSE on failure
                  or
                SM_TRUE on success
*/
IOPORTCONNECTION_API SM_BOOLEAN CALL_CONVENT BeginCheckedBlockEx(void * port, void * starPrinterStatus, UINT32 level);

/*
    EndCheckedBlockEx
    --------
    This function ends a checked block printing operation and returns the devices detailed status.
    This function does not return until either the printer has successfully printed all data or has gone offline in error.
    If the starPrinterStatus structure indicates that the printer is online upon return than all data was successfully printed.

    Parameters: port              - pointer to a previously created port
                starPrinterStatus - pointer to a StarPrinterStatus_n structure where the devices detailed status is written (structure is StarPrinterStatus_0)
                level             - integer designating the level of status structure (level is 0)

    Returns:    SM_FALSE on failure
                  or
                SM_TRUE on success
*/
IOPORTCONNECTION_API SM_BOOLEAN CALL_CONVENT EndCheckedBlockEx(void * port, void * starPrinterStatus, UINT32 level);

#ifdef __cplusplus
}
#endif

#endif

